﻿<#
.SYNOPSIS
  Antigravity Quota Monitor V2 — 高機能版クオータ確認スクリプト

.DESCRIPTION
  Antigravity のローカル Language Server から GetUserStatus を叩き、
  モデルごとの残量・リセット時刻を「視覚的に」表示します。
  V1 との違い:
    - プログレスバー（視覚的な残量表示）
    - リセットまでのカウントダウン（あと○時間○分）
    - 危険度に応じた色分け（緑→黄→赤）
    - Gemini系 / Vertex AI系（Claude等）のバケット分類
    - 実行ログの保存（log.json に追記し、前回との差分を表示）
    - 生レスポンスの全フィールドダンプ（--raw オプション）

.PARAMETER Raw
  -Raw を付けると、GetUserStatus の生レスポンスを JSON で出力します。

.EXAMPLE
  .\get_ag_quota.ps1
  .\get_ag_quota.ps1 -Raw

.NOTES
  作者: Midori Photo Art / 2026-03-01 V2
  前提: Windows、Antigravity 起動中
  証明書検証を無効化してローカル HTTPS に接続します。自己責任でご利用ください。
#>

param([switch]$Raw)

# ─── 色付きヘルパー ───
function Write-Color {
  param([string]$Text, [ConsoleColor]$Color = 'White')
  $prev = $Host.UI.RawUI.ForegroundColor
  $Host.UI.RawUI.ForegroundColor = $Color
  Write-Host $Text -NoNewline
  $Host.UI.RawUI.ForegroundColor = $prev
}

function Write-ColorLine {
  param([string]$Text, [ConsoleColor]$Color = 'White')
  Write-Color $Text $Color
  Write-Host ""
}

function Get-BarColor {
  param([double]$Pct)
  if ($Pct -ge 60) { return 'Green' }
  if ($Pct -ge 30) { return 'Yellow' }
  return 'Red'
}

function Format-Bar {
  param([double]$Pct, [int]$Width = 20)
  $filled = [math]::Round($Pct / 100 * $Width)
  if ($filled -gt $Width) { $filled = $Width }
  if ($filled -lt 0) { $filled = 0 }
  $empty = $Width - $filled
  $bar = ('#' * $filled) + ('-' * $empty)
  return $bar
}

function Format-Countdown {
  param([string]$ResetTimeStr)
  try {
    # resetTime は ISO8601 (UTC) or Unix timestamp
    if ($ResetTimeStr -match '^\d+$') {
      $resetUtc = [DateTimeOffset]::FromUnixTimeSeconds([long]$ResetTimeStr).UtcDateTime
    }
    else {
      # Parse して UTC に統一（"Z" 付きの場合ローカル変換されるのを防ぐ）
      $parsed = [DateTime]::Parse($ResetTimeStr)
      $resetUtc = $parsed.ToUniversalTime()
    }
    $now = [DateTime]::UtcNow
    $diff = $resetUtc - $now
    # リセット後の日本時間も計算
    $resetJst = $resetUtc.AddHours(9)
    $resetJstStr = $resetJst.ToString('HH:mm')
    if ($diff.TotalSeconds -le 0) { return "リセット済み ✓" }
    $h = [math]::Floor($diff.TotalHours)
    $m = $diff.Minutes
    return "あと ${h}時間${m}分 (${resetJstStr} JST)"
  }
  catch {
    return $ResetTimeStr
  }
}

function Get-Bucket {
  param([string]$Label)
  $lower = $Label.ToLower()
  if ($lower -match 'claude|opus|sonnet|gpt|vertex') { return 'Vertex AI (Claude/GPT)' }
  if ($lower -match 'gemini|flash') { return 'Gemini' }
  return 'Other'
}

# ─── ログ保存パス ───
$logDir = Split-Path -Parent $MyInvocation.MyCommand.Definition
$logFile = Join-Path $logDir 'quota_log.json'

# ─── Antigravity プロセス検出 ───
Write-Host ""
Write-ColorLine "═══════════════════════════════════════════════════" Cyan
Write-ColorLine "  Antigravity Quota Monitor V2  (2026-03-01)" Cyan
Write-ColorLine "═══════════════════════════════════════════════════" Cyan
Write-Host ""

$procs = Get-CimInstance Win32_Process | Where-Object { $_.Name -eq 'language_server_windows_x64.exe' -and $_.CommandLine -match 'antigravity' }
if (-not $procs -or $procs.Count -eq 0) {
  Write-ColorLine "✖ Antigravity プロセスが見つかりません。Antigravity を起動してから再実行してください。" Red
  exit 1
}
$proc = $procs[0]
$procPid = $proc.ProcessId
$cmd = $proc.CommandLine

Write-Color "  PID: " Gray; Write-ColorLine "$procPid" White

if ($cmd -match '--csrf_token[=\s]+([a-f0-9\-]+)') {
  $token = $Matches[1]
}
else {
  Write-ColorLine "✖ CSRF トークンが見つかりません。" Red
  exit 1
}

$portsFromNet = (Get-NetTCPConnection -OwningProcess $procPid -State Listen -ErrorAction SilentlyContinue | Select-Object -ExpandProperty LocalPort) | Sort-Object -Unique
if ($cmd -match '--extension_server_port[=\s]+(\d+)') {
  $extPort = [int]$Matches[1]
  $ports = @($extPort) + ($portsFromNet | Where-Object { $_ -ne $extPort })
}
else {
  $ports = @($portsFromNet)
}
if (-not $ports) {
  Write-ColorLine "✖ リッスンポートが見つかりません (PID $procPid)" Red
  exit 1
}
Write-Color "  Port候補: " Gray; Write-ColorLine "$($ports -join ', ')" White
Write-Host ""

# ─── API 呼び出し ───
[System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }
$body = '{"metadata":{"ideName":"antigravity","extensionName":"antigravity","locale":"en"}}'
$headers = @{ "X-Codeium-Csrf-Token" = $token; "Connect-Protocol-Version" = "1" }
$response = $null

foreach ($port in $ports) {
  try {
    $response = Invoke-RestMethod -Uri "https://127.0.0.1:$port/exa.language_server_pb.LanguageServerService/GetUserStatus" `
      -Method Post -Body $body -ContentType "application/json" -Headers $headers -TimeoutSec 5
    Write-Color "  接続成功: " Gray; Write-ColorLine "port $port" Green
    break
  }
  catch { }
}

if (-not $response) {
  Write-ColorLine "✖ GetUserStatus 全ポートで失敗しました。" Red
  exit 1
}

# ─── Raw モード ───
if ($Raw) {
  Write-Host ""
  Write-ColorLine "── 生レスポンス (JSON) ──" Yellow
  $response | ConvertTo-Json -Depth 20
  exit 0
}

# ─── userStatus の全体構造探索 ───
$userStatus = $response.userStatus
Write-Host ""
Write-ColorLine "── ユーザーステータス ──" Yellow

# プラン情報（planStatus / userTier）
$planStatus = $userStatus.planStatus
$planInfo = $planStatus.planInfo
$userTier = $userStatus.userTier

if ($userStatus.email) {
  Write-Color "  アカウント: " Gray; Write-ColorLine "$($userStatus.email)" White
}
if ($userTier -and $userTier.name) {
  Write-Color "  プラン: " Gray; Write-ColorLine "$($userTier.name)  ($($userTier.id))" Cyan
}
if ($planInfo -and $planInfo.planName) {
  Write-Color "  Plan Name: " Gray; Write-ColorLine "$($planInfo.planName)  (Tier: $($planInfo.teamsTier))" White
}

# ─── クレジット情報（重要！） ───
Write-Host ""
Write-ColorLine "── クレジット残量 ──" Yellow

if ($planInfo) {
  $monthlyPrompt = $planInfo.monthlyPromptCredits
  $monthlyFlow = $planInfo.monthlyFlowCredits
  $availPrompt = $planStatus.availablePromptCredits
  $availFlow = $planStatus.availableFlowCredits

  if ($monthlyPrompt -and $monthlyPrompt -gt 0) {
    $promptPct = [math]::Round($availPrompt / $monthlyPrompt * 100, 1)
    $promptBar = Format-Bar $promptPct
    $promptColor = Get-BarColor $promptPct
    Write-Color "  Prompt Credits:  " Gray
    Write-Color $promptBar $promptColor
    Write-Color " $($promptPct.ToString('0.0').PadLeft(5))%" $promptColor
    Write-ColorLine "   ($availPrompt / $monthlyPrompt)" DarkGray
  }

  if ($monthlyFlow -and $monthlyFlow -gt 0) {
    $flowPct = [math]::Round($availFlow / $monthlyFlow * 100, 1)
    $flowBar = Format-Bar $flowPct
    $flowColor = Get-BarColor $flowPct
    Write-Color "  Flow Credits:    " Gray
    Write-Color $flowBar $flowColor
    Write-Color " $($flowPct.ToString('0.0').PadLeft(5))%" $flowColor
    Write-ColorLine "   ($availFlow / $monthlyFlow)" DarkGray
  }

  if ($planInfo.monthlyFlexCreditPurchaseAmount) {
    Write-Color "  Flex購入枠: " Gray; Write-ColorLine "$($planInfo.monthlyFlexCreditPurchaseAmount) credits/月" DarkGray
  }
  if ($planInfo.canBuyMoreCredits) {
    Write-Color "  追加購入: " Gray; Write-ColorLine "可能" Green
  }
}

# アップグレード案内
if ($userTier -and $userTier.upgradeSubscriptionText) {
  Write-Host ""
  Write-Color "  Tip: " DarkYellow; Write-ColorLine "$($userTier.upgradeSubscriptionText)" DarkGray
}

# ─── モデル別クオータ ───
$models = $userStatus.cascadeModelConfigData.clientModelConfigs | Where-Object { $_.quotaInfo }
if (-not $models -or $models.Count -eq 0) {
  Write-ColorLine "  クオータ情報のあるモデルが見つかりません。" Yellow
  exit 0
}

Write-Host ""
Write-ColorLine "── モデル別クオータ ──" Yellow
Write-Host ""

# バケット分類
$buckets = @{}
foreach ($m in $models) {
  $bucket = Get-Bucket $m.label
  if (-not $buckets.ContainsKey($bucket)) { $buckets[$bucket] = @() }
  $buckets[$bucket] += $m
}

# 現在のスナップショット（ログ用）
$snapshot = @{ timestamp = (Get-Date -Format 'yyyy-MM-dd HH:mm:ss'); models = @() }

foreach ($bucketName in ($buckets.Keys | Sort-Object)) {
  Write-Color "  ┌─ " DarkGray; Write-ColorLine "$bucketName バケット" Cyan
  foreach ($m in $buckets[$bucketName]) {
    $pct = [math]::Round($m.quotaInfo.remainingFraction * 100, 1)
    $bar = Format-Bar $pct
    $barColor = Get-BarColor $pct
    $countdown = Format-Countdown $m.quotaInfo.resetTime
    $label = $m.label
    if ($label.Length -gt 28) { $label = $label.Substring(0, 25) + '...' }
    $label = $label.PadRight(28)

    Write-Color "  │ " DarkGray
    Write-Color "$label " White
    Write-Color $bar $barColor
    Write-Color " $($pct.ToString('0.0').PadLeft(5))% " $barColor
    if ($countdown -match 'リセット済み') {
      Write-ColorLine "  $countdown" Green
    }
    else {
      Write-ColorLine "  $countdown" Gray
    }

    # quotaInfo の追加フィールドを探索
    $qi = $m.quotaInfo
    $extra = @()
    if ($qi.dailyLimit) { $extra += "日次上限: $($qi.dailyLimit)" }
    if ($qi.weeklyLimit) { $extra += "週次上限: $($qi.weeklyLimit)" }
    if ($qi.usedTokens) { $extra += "消費トークン: $($qi.usedTokens)" }
    if ($qi.totalTokens) { $extra += "総トークン: $($qi.totalTokens)" }
    if ($qi.computeUsed) { $extra += "Compute消費: $($qi.computeUsed)" }
    if ($qi.promptCredits) { $extra += "Prompt Credits: $($qi.promptCredits)" }
    if ($extra.Count -gt 0) {
      Write-Color "  │   " DarkGray; Write-ColorLine "  $($extra -join ' / ')" DarkYellow
    }

    $snapshot.models += @{
      label     = $m.label
      bucket    = $bucketName
      remaining = $pct
      resetTime = "$($m.quotaInfo.resetTime)"
    }
  }
  Write-Color "  └" DarkGray; Write-ColorLine "─────────────────────────────────────────────" DarkGray
}

# ─── 前回との差分 ───
Write-Host ""
if (Test-Path $logFile) {
  try {
    $prevLogs = Get-Content $logFile -Raw | ConvertFrom-Json
    if ($prevLogs -and $prevLogs.Count -gt 0) {
      $lastLog = $prevLogs[-1]
      $elapsed = (Get-Date) - [DateTime]::Parse($lastLog.timestamp)
      $elapsedMin = [math]::Round($elapsed.TotalMinutes, 0)
      Write-ColorLine "── 前回からの変化（${elapsedMin}分前） ──" Yellow
      foreach ($cur in $snapshot.models) {
        $prev = $lastLog.models | Where-Object { $_.label -eq $cur.label }
        if ($prev) {
          $diff = [math]::Round($cur.remaining - $prev.remaining, 1)
          $diffStr = if ($diff -ge 0) { "+${diff}%" } else { "${diff}%" }
          $diffColor = if ($diff -ge 0) { 'Green' } else { 'Red' }
          $shortLabel = $cur.label
          if ($shortLabel.Length -gt 28) { $shortLabel = $shortLabel.Substring(0, 25) + '...' }
          Write-Color "  $($shortLabel.PadRight(28)) " Gray
          Write-Color "$($prev.remaining.ToString('0.0').PadLeft(5))%" Gray
          Write-Color " → " DarkGray
          Write-Color "$($cur.remaining.ToString('0.0').PadLeft(5))%" White
          Write-Color "  ( " DarkGray
          Write-Color $diffStr $diffColor
          Write-ColorLine " )" DarkGray
        }
      }
    }
  }
  catch { }
}

# ─── ログ保存 ───
try {
  $allLogs = @()
  if (Test-Path $logFile) {
    $existing = Get-Content $logFile -Raw | ConvertFrom-Json
    if ($existing) { $allLogs = @($existing) }
  }
  $allLogs += $snapshot
  # 最新50件のみ保持
  if ($allLogs.Count -gt 50) { $allLogs = $allLogs[($allLogs.Count - 50)..($allLogs.Count - 1)] }
  $allLogs | ConvertTo-Json -Depth 10 | Set-Content $logFile -Encoding UTF8
  Write-Host ""
  Write-Color "  📝 ログ保存: " Gray; Write-ColorLine $logFile DarkGray
}
catch {
  Write-Color "  ⚠ ログ保存失敗: " Yellow; Write-ColorLine $_.Exception.Message DarkGray
}

# ─── サマリー ───
Write-Host ""
$allPcts = $snapshot.models | ForEach-Object { $_.remaining }
$minPct = ($allPcts | Measure-Object -Minimum).Minimum
$avgPct = [math]::Round(($allPcts | Measure-Object -Average).Average, 1)
Write-Color "  📊 全体平均: " Gray; Write-Color "${avgPct}%" (Get-BarColor $avgPct)
Write-Color "  │  最低: " Gray; Write-ColorLine "${minPct}%" (Get-BarColor $minPct)

if ($minPct -lt 10) {
  Write-ColorLine "  🚨 警告: クオータがほぼ枯渇しています！重い操作は控えてください。" Red
}
elseif ($minPct -lt 30) {
  Write-ColorLine "  ⚠️  注意: クオータが減っています。Flash など軽量モデルの利用を推奨します。" Yellow
}
else {
  Write-ColorLine "  ✅ 余裕あり。通常通り利用できます。" Green
}

Write-Host ""
Write-ColorLine "═══════════════════════════════════════════════════" Cyan
Write-Host ""



