import * as vscode from 'vscode';

export function activate(context: vscode.ExtensionContext) {
    console.log('Extension "antigravity-clock-ext" is now active!');

    const provider = new ClockViewProvider(context.extensionUri);

    context.subscriptions.push(
        vscode.window.registerWebviewViewProvider(ClockViewProvider.viewType, provider)
    );
}

class ClockViewProvider implements vscode.WebviewViewProvider {
    public static readonly viewType = 'antigravity.clockView';

    constructor(
        private readonly _extensionUri: vscode.Uri,
    ) { }

    public resolveWebviewView(
        webviewView: vscode.WebviewView,
        context: vscode.WebviewViewResolveContext,
        _token: vscode.CancellationToken,
    ) {
        webviewView.webview.options = {
            // Webview内でスクリプトを実行できるようにする
            enableScripts: true,
        };

        webviewView.webview.html = this._getHtmlForWebview();
    }

    private _getHtmlForWebview() {
        return `<!DOCTYPE html>
<html lang="ja">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Clock & Calendar</title>
    <style>
        body {
            font-family: var(--vscode-font-family);
            color: var(--vscode-editor-foreground);
            background-color: var(--vscode-editor-background);
            display: flex;
            flex-direction: column;
            align-items: center;
            padding: 20px 10px;
            margin: 0;
            overflow: hidden;
        }

        /* アナログ時計のスタイル */
        .clock {
            position: relative;
            width: 150px;
            height: 150px;
            border-radius: 50%;
            border: 4px solid var(--vscode-focusBorder);
            background: var(--vscode-editorWidget-background);
            box-shadow: 0 0 15px rgba(0,0,0,0.5);
            margin-bottom: 30px;
            margin-top: 10px;
        }

        .clock::after {
            content: '';
            position: absolute;
            width: 10px;
            height: 10px;
            background: var(--vscode-focusBorder);
            border-radius: 50%;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            z-index: 10;
        }

        .hand {
            position: absolute;
            bottom: 50%;
            left: 50%;
            transform-origin: bottom;
            background: var(--vscode-editor-foreground);
            border-radius: 4px;
        }

        .hour-hand {
            width: 4px;
            height: 45px;
            margin-left: -2px;
        }

        .minute-hand {
            width: 3px;
            height: 60px;
            margin-left: -1.5px;
            background: var(--vscode-textLink-foreground);
        }

        .second-hand {
            width: 2px;
            height: 70px;
            margin-left: -1px;
            background: var(--vscode-errorForeground);
        }

        /* カレンダーのスタイル */
        .calendar-container {
            width: 100%;
            max-width: 220px;
            text-align: center;
        }

        .calendar-header {
            font-size: 1.2em;
            font-weight: bold;
            margin-bottom: 15px;
            color: var(--vscode-textLink-foreground);
        }

        table {
            width: 100%;
            border-collapse: collapse;
            table-layout: fixed;
        }

        th, td {
            text-align: center;
            padding: 6px 4px;
            font-size: 0.9em;
        }

        th {
            color: var(--vscode-descriptionForeground);
            border-bottom: 1px solid var(--vscode-widget-border);
            padding-bottom: 8px;
            font-weight: normal;
        }

        td {
            border-radius: 4px;
            cursor: default;
        }

        td:hover {
            background: var(--vscode-list-hoverBackground);
        }

        .today {
            background: var(--vscode-focusBorder);
            color: var(--vscode-button-foreground);
            font-weight: bold;
        }
        .today:hover {
             background: var(--vscode-focusBorder);
        }
    </style>
</head>
<body>
    <div class="clock">
        <div class="hand hour-hand" id="hour"></div>
        <div class="hand minute-hand" id="minute"></div>
        <div class="hand second-hand" id="second"></div>
    </div>

    <div class="calendar-container">
        <div class="calendar-header" id="month-year"></div>
        <table id="calendar">
            <thead>
                <tr>
                    <th>S</th><th>M</th><th>T</th><th>W</th><th>T</th><th>F</th><th>S</th>
                </tr>
            </thead>
            <tbody id="calendar-body">
            </tbody>
        </table>
    </div>

    <script>
        // --- 時計の実装 ---
        const hourHand = document.getElementById('hour');
        const minuteHand = document.getElementById('minute');
        const secondHand = document.getElementById('second');

        function updateClock() {
            const now = new Date();
            const hours = now.getHours();
            const minutes = now.getMinutes();
            const seconds = now.getSeconds();

            // 針の角度を計算
            const hourDeg = (hours % 12) * 30 + minutes * 0.5;
            const minuteDeg = minutes * 6 + seconds * 0.1;
            const secondDeg = seconds * 6;

            hourHand.style.transform = \`translateX(-50%) rotate(\${hourDeg}deg)\`;
            minuteHand.style.transform = \`translateX(-50%) rotate(\${minuteDeg}deg)\`;
            secondHand.style.transform = \`translateX(-50%) rotate(\${secondDeg}deg)\`;
        }
        
        setInterval(updateClock, 1000);
        updateClock(); // 初期描画

        // --- カレンダーの実装 ---
        function renderCalendar() {
            const now = new Date();
            const currentMonth = now.getMonth();
            const currentYear = now.getFullYear();
            const today = now.getDate();

            document.getElementById('month-year').textContent = currentYear + ' / ' + (currentMonth + 1);

            const firstDay = new Date(currentYear, currentMonth, 1).getDay();
            const daysInMonth = new Date(currentYear, currentMonth + 1, 0).getDate();

            const tbody = document.getElementById('calendar-body');
            tbody.innerHTML = ''; // 中身をクリア

            let date = 1;
            for (let i = 0; i < 6; i++) {
                const row = document.createElement('tr');
                for (let j = 0; j < 7; j++) {
                    const cell = document.createElement('td');
                    if (i === 0 && j < firstDay) {
                        cell.textContent = '';
                    } else if (date > daysInMonth) {
                        break;
                    } else {
                        cell.textContent = date;
                        if (date === today) {
                            cell.classList.add('today');
                        }
                        date++;
                    }
                    row.appendChild(cell);
                }
                tbody.appendChild(row);
                if (date > daysInMonth) break;
            }
        }
        
        renderCalendar();
    </script>
</body>
</html>`;
    }
}

export function deactivate() { }
