// --- Elements ---
const urlInput = document.getElementById('url-input');
const iframe = document.getElementById('browser-frame');
const backBtn = document.getElementById('back-btn');
const forwardBtn = document.getElementById('forward-btn');
const refreshBtn = document.getElementById('refresh-btn');
const inspectBtn = document.getElementById('inspect-mode-btn');
const syncBtn = document.getElementById('ai-sync-btn');
const sidePanelToggle = document.getElementById('side-panel-toggle');
const sidePanel = document.getElementById('side-panel');
const overlay = document.getElementById('inspector-overlay');
const highlight = document.getElementById('highlight-box');
const marginOverlay = document.getElementById('margin-overlay');
const paddingOverlay = document.getElementById('padding-overlay');
const tagLabel = document.getElementById('tag-label');

const tabBtns = document.querySelectorAll('.tab-btn');
const tabPanes = document.querySelectorAll('.tab-pane');

const categoryMap = {
    layout: ['div', 'section', 'header', 'main', 'footer', 'aside', 'nav', 'article', 'body'],
    typography: ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'p', 'span', 'a', 'strong', 'em', 'b', 'i', 'blockquote', 'li', 'ul', 'ol'],
    media: ['img', 'video', 'audio', 'svg', 'canvas', 'iframe', 'picture', 'source'],
    form: ['form', 'input', 'button', 'select', 'textarea', 'label', 'fieldset', 'legend', 'option'],
    table: ['table', 'tr', 'td', 'th', 'tbody', 'thead', 'tfoot', 'colgroup', 'col']
};

function getElementCategory(tagName) {
    const t = tagName.toLowerCase();
    for (const [category, tags] of Object.entries(categoryMap)) {
        if (tags.includes(t)) return category;
    }
    return 'default';
}

let lastSelectedElement = null;

// --- Tab Switching ---
tabBtns.forEach(btn => {
    btn.addEventListener('click', () => {
        const tabName = btn.getAttribute('data-tab');
        tabBtns.forEach(b => b.classList.remove('active'));
        tabPanes.forEach(p => p.classList.remove('active'));
        btn.classList.add('active');
        document.getElementById(`tab-${tabName}`).classList.add('active');
    });
});

sidePanelToggle.addEventListener('click', () => {
    sidePanel.classList.toggle('hidden');
});

urlInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
        let url = urlInput.value;
        if (!url.startsWith('http') && !url.includes('.html')) {
            url = `https://www.google.com/search?q=${encodeURIComponent(url)}`;
        } else if (!url.startsWith('http') && !url.startsWith('../')) {
            // ローカルファイル名が直接入力された場合、親フォルダ基準に変換
            url = '../' + url;
        }
        iframe.src = url;
    }
});

refreshBtn.addEventListener('click', () => {
    iframe.contentWindow.location.reload();
});

// Update Address Bar when iframe navigates (clicking links, back/forward)
iframe.addEventListener('load', () => {
    try {
        const currentUrl = iframe.contentWindow.location.href;
        // Clean up the URL for display if it's a local file
        if (currentUrl.startsWith('file:///')) {
            const parts = currentUrl.split('/');
            // Just show the filename or last two parts to keep it clean
            urlInput.value = parts.slice(-2).join('/');
        } else {
            urlInput.value = currentUrl;
        }
    } catch (e) {
        console.warn('Cannot read iframe URL due to cross-origin restrictions.');
    }
});

backBtn.addEventListener('click', () => {
    try {
        iframe.contentWindow.history.back();
    } catch (e) {
        console.warn('Could not navigate back:', e);
    }
});

forwardBtn.addEventListener('click', () => {
    try {
        iframe.contentWindow.history.forward();
    } catch (e) {
        console.warn('Could not navigate forward:', e);
    }
});

// --- Inspector Engine ---
inspectBtn.addEventListener('click', () => {
    inspectBtn.classList.toggle('active');
    overlay.classList.toggle('active');
    if (!overlay.classList.contains('active')) {
        highlight.style.display = 'none';
        marginOverlay.style.display = 'none';
        paddingOverlay.style.display = 'none';
    }
});

overlay.addEventListener('mousemove', (e) => {
    if (!inspectBtn.classList.contains('active')) return;

    try {
        const frameWin = iframe.contentWindow;
        const frameDoc = iframe.contentDocument || frameWin.document;
        const rect = iframe.getBoundingClientRect();

        const x = e.clientX - rect.left;
        const y = e.clientY - rect.top;

        const el = frameDoc.elementFromPoint(x, y);
        if (el && el !== frameDoc.documentElement && el !== frameDoc.body) {
            updateInspection(el, rect);
        }
    } catch (err) {
        console.warn("Cross-origin restriction or Iframe not ready");
    }
});

overlay.addEventListener('click', () => {
    if (inspectBtn.classList.contains('active')) {
        // Lock exploration and show panel
        sidePanel.classList.remove('hidden');
        inspectBtn.classList.remove('active');
        overlay.classList.remove('active');
    }
});

function updateInspection(el, frameRect) {
    lastSelectedElement = el;
    const elRect = el.getBoundingClientRect();
    const style = window.getComputedStyle(el);

    // 1. Highlight Overlay (Content Box)
    const category = getElementCategory(el.tagName);
    highlight.className = 'highlight-box active';
    highlight.classList.add(`theme-${category}`);
    highlight.style.width = `${elRect.width}px`;
    highlight.style.height = `${elRect.height}px`;
    highlight.style.left = `${elRect.left}px`;
    highlight.style.top = `${elRect.top}px`;
    highlight.style.display = 'block';

    // Rich Tag Label formatting
    const idHtml = el.id ? `<span class="id">#${el.id}</span>` : '';
    const classesHtml = Array.from(el.classList).map(c => `<span class="class">.${c}</span>`).join('');
    tagLabel.innerHTML = `<span class="tag">${el.tagName.toLowerCase()}</span>${idHtml}${classesHtml}`;

    // Margin Overlay Position calculation
    const mt = parseFloat(style.marginTop) || 0;
    const mr = parseFloat(style.marginRight) || 0;
    const mb = parseFloat(style.marginBottom) || 0;
    const ml = parseFloat(style.marginLeft) || 0;

    marginOverlay.style.width = `${elRect.width + ml + mr}px`;
    marginOverlay.style.height = `${elRect.height + mt + mb}px`;
    marginOverlay.style.left = `${elRect.left - ml}px`;
    marginOverlay.style.top = `${elRect.top - mt}px`;
    marginOverlay.style.display = 'block';

    // Padding Overlay Position calculation
    const pt = parseFloat(style.paddingTop) || 0;
    const pr = parseFloat(style.paddingRight) || 0;
    const pb = parseFloat(style.paddingBottom) || 0;
    const pl = parseFloat(style.paddingLeft) || 0;
    const bt = parseFloat(style.borderTopWidth) || 0;
    const bl = parseFloat(style.borderLeftWidth) || 0;

    paddingOverlay.style.width = `${elRect.width - (parseFloat(style.paddingLeft) ? 0 : 0)}px`; // Adjust logic for box-sizing later if needed
    // Assuming border-box or adjusting relative to elRect directly
    paddingOverlay.style.width = `${elRect.width - bt * 2}px`;
    paddingOverlay.style.height = `${elRect.height - bl * 2}px`;

    paddingOverlay.style.width = `${elRect.width}px`;
    paddingOverlay.style.height = `${elRect.height}px`;
    paddingOverlay.style.left = `${elRect.left}px`;
    paddingOverlay.style.top = `${elRect.top}px`;

    // To properly show padding INSIDE the element
    paddingOverlay.style.width = `${elRect.width - bt * 2}px`;
    paddingOverlay.style.height = `${elRect.height - bl * 2}px`;
    paddingOverlay.style.borderTop = `${pt}px solid var(--color-padding)`;
    paddingOverlay.style.borderRight = `${pr}px solid var(--color-padding)`;
    paddingOverlay.style.borderBottom = `${pb}px solid var(--color-padding)`;
    paddingOverlay.style.borderLeft = `${pl}px solid var(--color-padding)`;
    paddingOverlay.style.background = 'transparent';
    paddingOverlay.style.display = 'block';

    // 2. Styles Tab
    document.getElementById('spec-selector').innerHTML = tagLabel.innerHTML;
    document.getElementById('spec-element').innerText = el.outerHTML.split('>')[0] + '>';
    document.getElementById('spec-bg').textContent = style.backgroundColor;
    document.getElementById('spec-font').textContent = `${style.fontSize} ${style.fontFamily.split(',')[0]} (${style.color})`;

    // 3. Layout Tab (Box Model)
    document.getElementById('box-content-size').textContent = `${Math.round(elRect.width)} x ${Math.round(elRect.height)}`;

    // Values constructed from computed style
    document.getElementById('spec-margin').textContent = `${style.marginTop} ${style.marginRight} ${style.marginBottom} ${style.marginLeft}`.replace(/(^| )0px/g, '$10');
    document.getElementById('spec-border').textContent = `${style.borderTopWidth} ${style.borderRightWidth} ${style.borderBottomWidth} ${style.borderLeftWidth}`.replace(/(^| )0px/g, '$10');
    document.getElementById('spec-padding').textContent = `${style.paddingTop} ${style.paddingRight} ${style.paddingBottom} ${style.paddingLeft}`.replace(/(^| )0px/g, '$10');

    document.getElementById('spec-display').textContent = style.display;
    document.getElementById('spec-position').textContent = style.position;

    // 4. Code Tab (Snippet Generation)
    generateSnippets(el);
}

function getSelectorPath(el) {
    const path = [];
    while (el.nodeType === Node.ELEMENT_NODE) {
        let selector = el.nodeName.toLowerCase();
        if (el.id) {
            selector += '#' + el.id;
            path.unshift(selector);
            break;
        } else {
            let sib = el, nth = 1;
            while (sib = sib.previousElementSibling) {
                if (sib.nodeName.toLowerCase() == selector) nth++;
            }
            if (nth != 1) selector += ":nth-of-type(" + nth + ")";
        }
        path.unshift(selector);
        el = el.parentNode;
    }
    return path.join(" > ");
}

function formatHTML(htmlStr) {
    let indentLevel = 0;
    const tabInfo = '  ';
    let text = htmlStr.replace(/>\s+</g, '><');
    text = text.replace(/</g, '~::~<').replace(/>/g, '>~::~');
    const lines = text.split('~::~');

    let result = '';
    for (let i = 0; i < lines.length; i++) {
        let line = lines[i].trim();
        if (!line) continue;

        if (line.match(/^<\//)) {
            indentLevel--;
            result += tabInfo.repeat(Math.max(0, indentLevel)) + line + '\n';
        } else if (line.match(/^<[a-zA-Z0-9\-]+/)) {
            result += tabInfo.repeat(Math.max(0, indentLevel)) + line + '\n';
            if (!line.match(/\/>$/) && !line.match(/^<(img|hr|br|input|meta|link|source)( |>)/i)) {
                indentLevel++;
            }
        } else {
            result += tabInfo.repeat(Math.max(0, indentLevel)) + line + '\n';
        }
    }
    return result.trim();
}

function highlightHTML(htmlStr) {
    return htmlStr
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/(&lt;\/?)([a-zA-Z0-9\-]+)/g, '$1<span class="token-tag">$2</span>')
        .replace(/([a-zA-Z0-9\-]+)=/g, '<span class="token-attr">$1</span>=')
        .replace(/"([^"]*)"/g, '<span class="token-val">"$1"</span>');
}

function highlightCSS(cssStr) {
    return cssStr
        .replace(/([a-zA-Z0-9\-]+)\s*:/g, '<span class="token-attr">$1</span>:')
        .replace(/:\s*([^;]+);/g, ': <span class="token-val">$1</span>;')
        .replace(/(\.[a-zA-Z0-9\-_]+)/g, '<span class="token-class">$1</span>');
}

function generateSnippets(el) {
    // Basic HTML Snippet with highlighting
    let tagContent = formatHTML(el.outerHTML);

    // 中身が長すぎる場合のみ省略する（例: 1500文字以上）
    if (tagContent.length > 1500) {
        tagContent = tagContent.substring(0, 1500) + '\n\n... <!-- HTMLが見やすさのため一部省略されました -->';
    }

    document.getElementById('code-snippet-html').innerHTML = highlightHTML(tagContent);

    // Clean CSS Snippet with highlighting
    const style = window.getComputedStyle(el);
    const importantProps = [
        'background-color', 'color', 'font-size', 'font-weight',
        'padding', 'margin', 'border-radius', 'box-shadow', 'display'
    ];
    let cssSnippet = `.selected-element {\n`;
    importantProps.forEach(prop => {
        const val = style.getPropertyValue(prop);
        if (val && val !== 'none' && val !== 'rgba(0, 0, 0, 0)' && val !== 'normal') {
            cssSnippet += `    ${prop}: ${val};\n`;
        }
    });
    cssSnippet += `}`;
    document.getElementById('code-snippet-css').innerHTML = highlightCSS(cssSnippet);
}

// --- Sync / Clipboard Auto-Copy ---
syncBtn.addEventListener('click', async () => {
    if (!lastSelectedElement) {
        alert("Please select an element first!");
        return;
    }

    const el = lastSelectedElement;
    const style = window.getComputedStyle(el);
    const elRect = el.getBoundingClientRect();

    // Generate data directly
    const data = {
        element: el.tagName.toLowerCase(),
        id: el.id,
        classes: Array.from(el.classList),
        dimensions: { width: elRect.width, height: elRect.height },
        styles: {
            color: style.color,
            background: style.backgroundColor,
            font: style.fontFamily,
            padding: style.padding,
            margin: style.margin
        },
        path: getSelectorPath(el)
    };

    const copyData = JSON.stringify(data, null, 2);

    const showCopiedIndicator = () => {
        syncBtn.innerHTML = '<i data-lucide="clipboard-check"></i> Copied!';
        setTimeout(() => {
            syncBtn.innerHTML = '<i data-lucide="copy"></i> Copy Element Data';
            lucide.createIcons();
        }, 3000);
    };

    // Robust copy function (Fallback for Webviews/Iframes)
    const copyToClipboard = async (text) => {
        if (navigator.clipboard && window.isSecureContext) {
            try {
                await navigator.clipboard.writeText(text);
                return true;
            } catch (err) {
                console.warn("navigator.clipboard failed, using fallback.", err);
            }
        }

        // Fallback method using textarea
        try {
            const textarea = document.createElement('textarea');
            textarea.value = text;
            textarea.style.position = 'fixed';
            textarea.style.opacity = '0';
            document.body.appendChild(textarea);
            textarea.select();
            const successful = document.execCommand('copy');
            document.body.removeChild(textarea);
            return successful;
        } catch (err) {
            console.error("Fallback clipboard failed:", err);
            return false;
        }
    };

    const success = await copyToClipboard(copyData);

    if (success) {
        showCopiedIndicator();
    } else {
        alert("Error: Clipboard access denied. Could not copy data.");
    }
});

// --- Sync & Auto-Reload Engine ---
let currentFileInKaraage = '';

async function checkSyncFile() {
    try {
        const response = await fetch('./.current_file.txt');
        if (response.ok) {
            let targetName = await response.text();
            targetName = targetName.trim();

            // ファイル名が変わったときに自動で切り替える
            if (targetName && targetName !== currentFileInKaraage) {
                currentFileInKaraage = targetName;

                let targetUrl = targetName;
                if (!targetUrl.startsWith('http') && !targetUrl.startsWith('../')) {
                    targetUrl = '../' + targetUrl; // プロジェクトルートからの相対パスへ
                }

                iframe.src = targetUrl;
                urlInput.value = targetName;
            }
        }
    } catch (e) {
        // bridge file not found, silently ignore
    }
}

// 初期化時と、1秒ごとに定期チェック
window.addEventListener('load', () => {
    checkSyncFile();
    setInterval(checkSyncFile, 1000); // ✨Auto-Sync Polling✨
});
