import os
import logging
from fastapi import FastAPI
from fastapi.responses import HTMLResponse

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = FastAPI(title="Overkill Hello")
GREETING = os.environ.get("GREETING", "Hello, World")


def _html_page() -> str:
    return f"""<!DOCTYPE html>
<html lang="ja">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Overkill Hello | Premium Edition</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Cinzel:wght@400;700&family=Playfair+Display:ital,wght@0,400;0,700;1,400&family=Zen+Old+Mincho:wght@400;700&display=swap" rel="stylesheet">
  <style>
    :root {{
      --gold-primary: #d4af37;
      --gold-light: #f3e5ab;
      --gold-dark: #aa8c2c;
      --bg-dark: #0f0f13;
      --bg-panel: rgba(20, 20, 25, 0.7);
    }}
    * {{ margin: 0; padding: 0; box-sizing: border-box; }}
    
    body {{
      min-height: 100vh;
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      background-color: var(--bg-dark);
      background-image: 
        radial-gradient(circle at 20% 30%, rgba(60, 20, 60, 0.4) 0%, transparent 50%),
        radial-gradient(circle at 80% 70%, rgba(20, 40, 60, 0.4) 0%, transparent 50%);
      color: #e2e8f0;
      font-family: 'Zen Old Mincho', serif;
      overflow: hidden;
    }}

    /* Animated background particles */
    .bg-particles {{
      position: absolute;
      top: 0; left: 0; width: 100%; height: 100%;
      pointer-events: none;
      z-index: -1;
      background: url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23d4af37' fill-opacity='0.05'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E");
    }}

    .container {{
      position: relative;
      padding: 4rem 3rem;
      background: var(--bg-panel);
      backdrop-filter: blur(20px);
      -webkit-backdrop-filter: blur(20px);
      border: 1px solid rgba(212, 175, 55, 0.2);
      border-radius: 4px;
      box-shadow: 
        0 20px 50px rgba(0,0,0,0.5),
        inset 0 0 0 1px rgba(255,255,255,0.05);
      text-align: center;
      max-width: 600px;
      width: 90%;
      animation: float 6s ease-in-out infinite;
    }}

    /* Decorative corners */
    .container::before, .container::after {{
      content: '';
      position: absolute;
      width: 20px;
      height: 20px;
      border: 2px solid var(--gold-primary);
      transition: all 0.5s ease;
    }}
    .container::before {{ top: -2px; left: -2px; border-right: none; border-bottom: none; }}
    .container::after {{ bottom: -2px; right: -2px; border-left: none; border-top: none; }}

    h1 {{
      font-family: 'Cinzel', serif;
      font-size: clamp(3rem, 6vw, 5rem);
      font-weight: 400;
      margin-bottom: 0.5rem;
      background: linear-gradient(to right, var(--gold-light), var(--gold-primary), var(--gold-light));
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-size: 200% auto;
      animation: shine 5s linear infinite;
      letter-spacing: 0.05em;
    }}

    .subtitle {{
      font-family: 'Playfair Display', serif;
      font-style: italic;
      color: rgba(255, 255, 255, 0.7);
      font-size: 1.2rem;
      margin-bottom: 3rem;
      letter-spacing: 0.1em;
    }}

    .badge-container {{
      margin-bottom: 3rem;
    }}

    .badge {{
      display: inline-block;
      padding: 0.5rem 1.5rem;
      border: 1px solid var(--gold-dark);
      color: var(--gold-primary);
      font-size: 0.8rem;
      text-transform: uppercase;
      letter-spacing: 0.2rem;
      transition: all 0.3s ease;
      position: relative;
      overflow: hidden;
    }}
    
    .badge::before {{
      content: '';
      position: absolute;
      top: 0; left: -100%;
      width: 100%; height: 100%;
      background: linear-gradient(90deg, transparent, rgba(212, 175, 55, 0.2), transparent);
      transition: all 0.5s;
    }}

    .badge:hover::before {{ left: 100%; }}
    .badge:hover {{
      background: rgba(212, 175, 55, 0.1);
      box-shadow: 0 0 15px rgba(212, 175, 55, 0.3);
    }}

    .links {{
      display: flex;
      justify-content: center;
      gap: 2rem;
      border-top: 1px solid rgba(255,255,255,0.1);
      padding-top: 2rem;
    }}

    .links a {{
      color: rgba(255,255,255,0.6);
      text-decoration: none;
      font-family: 'Cinzel', serif;
      font-size: 0.9rem;
      letter-spacing: 0.1rem;
      transition: all 0.3s ease;
      position: relative;
    }}

    .links a::after {{
      content: '';
      position: absolute;
      width: 0;
      height: 1px;
      bottom: -4px;
      left: 0;
      background-color: var(--gold-primary);
      transition: width 0.3s ease;
    }}

    .links a:hover {{
      color: var(--gold-light);
    }}

    .links a:hover::after {{
      width: 100%;
    }}

    @keyframes shine {{
      to {{ background-position: 200% center; }}
    }}

    @keyframes float {{
      0%, 100% {{ transform: translateY(0); }}
      50% {{ transform: translateY(-10px); }}
    }}

    @media (max-width: 600px) {{
      .container {{ padding: 2rem 1.5rem; }}
      .links {{ flex-direction: column; gap: 1rem; }}
    }}
  </style>
</head>
<body>
  <div class="bg-particles"></div>
  <div class="container">
    <h1>{GREETING}</h1>
    <p class="subtitle">Exquisite Cloud Run Experience</p>
    
    <div class="badge-container">
      <span class="badge">Edition v1.0.0</span>
    </div>

    <div class="links">
      <a href="/health">SYSTEM STATUS</a>
      <a href="/version">VERSION INFO</a>
      <a href="/docs" target="_blank">DOCUMENTATION</a>
    </div>
  </div>
</body>
</html>"""


@app.get("/", response_class=HTMLResponse)
def root():
    logger.info("root requested")
    return _html_page()


@app.get("/api")
def root_json():
    """JSON のみ欲しい場合は /api"""
    return {"message": GREETING}


@app.get("/health")
def health():
    return {"status": "ok"}


@app.get("/version")
def version():
    return {"version": "1.0.0", "app": "cloudrun_hello"}
