import os
import time
import logging
from fastapi import FastAPI, Query
from fastapi.responses import PlainTextResponse, HTMLResponse
import google.generativeai as genai

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)

app = FastAPI(title="Overkill Ask API")
API_KEY = os.environ.get("GEMINI_API_KEY", "")


def _html_page() -> str:
    return """<!DOCTYPE html>
<html lang="ja">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">
  <title>Overkill Ask | Premium AI Experience</title>
  <link rel="preconnect" href="https://fonts.googleapis.com">
  <link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
  <link href="https://fonts.googleapis.com/css2?family=Cinzel:wght@400;700&family=Playfair+Display:ital,wght@0,400;0,700;1,400&family=Zen+Old+Mincho:wght@400;700&display=swap" rel="stylesheet">
  <style>
    :root {
      --gold-primary: #d4af37;
      --gold-light: #f3e5ab;
      --gold-dark: #aa8c2c;
      --bg-dark: #0f0f13;
      --bg-panel: rgba(20, 20, 25, 0.85);
      --text-main: #e2e8f0;
    }
    * { margin: 0; padding: 0; box-sizing: border-box; }
    
    body {
      min-height: 100vh;
      display: flex;
      flex-direction: column;
      align-items: center;
      justify-content: center;
      background-color: var(--bg-dark);
      background-image: 
        radial-gradient(circle at 20% 30%, rgba(60, 20, 60, 0.4) 0%, transparent 50%),
        radial-gradient(circle at 80% 70%, rgba(20, 40, 60, 0.4) 0%, transparent 50%);
      color: var(--text-main);
      font-family: 'Zen Old Mincho', serif;
      overflow-x: hidden;
    }

    .bg-particles {
      position: absolute;
      top: 0; left: 0; width: 100%; height: 100%;
      pointer-events: none;
      z-index: -1;
      background: url("data:image/svg+xml,%3Csvg width='60' height='60' viewBox='0 0 60 60' xmlns='http://www.w3.org/2000/svg'%3E%3Cg fill='none' fill-rule='evenodd'%3E%3Cg fill='%23d4af37' fill-opacity='0.05'%3E%3Cpath d='M36 34v-4h-2v4h-4v2h4v4h2v-4h4v-2h-4zm0-30V0h-2v4h-4v2h4v4h2V6h4V4h-4zM6 34v-4H4v4H0v2h4v4h2v-4h4v-2H6zM6 4V0H4v4H0v2h4v4h2V6h4V4H6z'/%3E%3C/g%3E%3C/g%3E%3C/svg%3E");
    }

    .container {
      position: relative;
      padding: 3rem;
      background: var(--bg-panel);
      backdrop-filter: blur(20px);
      -webkit-backdrop-filter: blur(20px);
      border: 1px solid rgba(212, 175, 55, 0.2);
      border-radius: 8px;
      box-shadow: 0 20px 50px rgba(0,0,0,0.5);
      width: 90%;
      max-width: 800px;
      text-align: center;
      transition: all 0.5s ease;
    }

    /* Decorative corners */
    .container::before, .container::after {
      content: '';
      position: absolute;
      width: 30px;
      height: 30px;
      border: 2px solid var(--gold-primary);
      transition: all 0.5s ease;
    }
    .container::before { top: -2px; left: -2px; border-right: none; border-bottom: none; }
    .container::after { bottom: -2px; right: -2px; border-left: none; border-top: none; }

    h1 {
      font-family: 'Cinzel', serif;
      font-size: clamp(2rem, 5vw, 3.5rem);
      font-weight: 400;
      margin-bottom: 0.5rem;
      background: linear-gradient(to right, var(--gold-light), var(--gold-primary), var(--gold-light));
      -webkit-background-clip: text;
      -webkit-text-fill-color: transparent;
      background-size: 200% auto;
      animation: shine 5s linear infinite;
    }

    p.subtitle {
      font-family: 'Playfair Display', serif;
      font-style: italic;
      color: rgba(255, 255, 255, 0.6);
      margin-bottom: 2rem;
    }

    .input-group {
      position: relative;
      margin-bottom: 2rem;
    }

    input[type="text"] {
      width: 100%;
      background: rgba(255,255,255,0.05);
      border: 1px solid rgba(255,255,255,0.1);
      border-radius: 4px;
      padding: 1rem 1.5rem;
      color: #fff;
      font-family: 'Zen Old Mincho', serif;
      font-size: 1.1rem;
      transition: all 0.3s;
      outline: none;
    }

    input[type="text"]:focus {
      border-color: var(--gold-primary);
      box-shadow: 0 0 15px rgba(212, 175, 55, 0.1);
      background: rgba(255,255,255,0.08);
    }

    button {
      margin-top: 1rem;
      background: transparent;
      color: var(--gold-primary);
      border: 1px solid var(--gold-primary);
      padding: 0.8rem 2.5rem;
      font-family: 'Cinzel', serif;
      font-size: 1rem;
      cursor: pointer;
      transition: all 0.3s;
      letter-spacing: 0.1em;
      position: relative;
      overflow: hidden;
    }

    button:hover {
      background: var(--gold-primary);
      color: var(--bg-dark);
      box-shadow: 0 0 20px rgba(212, 175, 55, 0.4);
    }
    
    button:disabled {
      opacity: 0.5;
      cursor: not-allowed;
    }

    #response-area {
      margin-top: 2rem;
      text-align: left;
      min-height: 100px;
      padding: 1.5rem;
      background: rgba(0,0,0,0.2);
      border-left: 2px solid var(--gold-dark);
      opacity: 0;
      transform: translateY(10px);
      transition: all 0.5s ease;
      display: none; /* Hidden initially */
    }

    #response-area.visible {
      opacity: 1;
      transform: translateY(0);
      display: block;
    }

    .loading-pulse {
      display: none;
      width: 40px;
      height: 40px;
      margin: 2rem auto;
      border: 2px solid var(--gold-primary);
      border-radius: 50%;
      animation: pulse 1.5s infinite;
    }

    .error-msg { color: #ff6b6b; }

    @keyframes shine { to { background-position: 200% center; } }
    @keyframes pulse {
      0% { transform: scale(0.9); opacity: 0.7; }
      50% { transform: scale(1.1); opacity: 1; box-shadow: 0 0 15px var(--gold-primary); }
      100% { transform: scale(0.9); opacity: 0.7; }
    }
  </style>
</head>
<body>
  <div class="bg-particles"></div>
  <div class="container">
    <h1>Oraculum AI</h1>
    <p class="subtitle">Ask and the void shall answer...</p>

    <div class="input-group">
      <input type="text" id="question" placeholder="Enter your query here..." maxlength="100">
    </div>
    
    <button onclick="askOracle()" id="ask-btn">INQUIRE</button>
    
    <div class="loading-pulse" id="loader"></div>
    
    <div id="response-area"></div>
  </div>

  <script>
    async function askOracle() {
      const qInput = document.getElementById('question');
      const btn = document.getElementById('ask-btn');
      const loader = document.getElementById('loader');
      const responseArea = document.getElementById('response-area');
      
      const question = qInput.value.trim();
      if (!question) return;

      // Reset UI
      responseArea.classList.remove('visible');
      responseArea.innerHTML = '';
      btn.disabled = true;
      loader.style.display = 'block';

      try {
        const res = await fetch(`/ask?q=${encodeURIComponent(question)}`);
        const text = await res.text();
        
        loader.style.display = 'none';
        responseArea.innerHTML = text; // Plain text response
        responseArea.classList.add('visible');
      } catch (e) {
        loader.style.display = 'none';
        responseArea.innerHTML = '<span class="error-msg">The oracle is silent... (Connection Error)</span>';
        responseArea.classList.add('visible');
      } finally {
        btn.disabled = false;
        qInput.focus();
      }
    }

    // Enter key support
    document.getElementById('question').addEventListener('keypress', function (e) {
      if (e.key === 'Enter') askOracle();
    });
  </script>
</body>
</html>
"""


@app.get("/", response_class=HTMLResponse)
def root():
    return _html_page()


@app.get("/ask", response_class=PlainTextResponse)
def ask(q: str = Query(..., min_length=1, max_length=500)):
    if not API_KEY:
        return "GEMINI_API_KEY が未設定です。"
    
    genai.configure(api_key=API_KEY)
    # Using 1.5-flash for better stability
    model = genai.GenerativeModel("gemini-2.5-flash")

    max_retries = 3
    for attempt in range(max_retries):
        try:
            r = model.generate_content(q[:500])
            return r.text or "(空の応答)"
        except Exception as e:
            if "429" in str(e) and attempt < max_retries - 1:
                time.sleep(2 * (attempt + 1))  # Exponential backoff
                continue
            return f"Error: {e}"
