#!/usr/bin/env node
/**
 * unidic-converter.js
 * UniDic の lex.csv から MorphFoundry 用 JSON 辞書を生成するユーティリティ
 *
 * 使い方:
 *   node unidic-converter.js /path/to/lex.csv ./dictionaries/unidic-mini.json 50000
 *
 * 第3引数はオプションで、何件のエントリを取り込むかを制限できます。
 * （指定しない場合は全件を読み込みますが、数百万行になるので注意してください）
 *
 * 出力される JSON は `label` に mini/full を含むメタ情報つきで保存されます。
 * cost は UniDic に含まれないため、デフォルト値 5000 を与えています。
 * 必要に応じて MorphFoundry 側で connectionCost / unknownCost と併せて調整してください。
 */

import fs from 'fs';
import path from 'path';
import readline from 'readline';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

async function main() {
  const [, , inputPathArg, outputPathArg, limitArg] = process.argv;

  if (!inputPathArg || !outputPathArg) {
    console.error('使用方法: node unidic-converter.js <lex.csvへのパス> <出力先JSON> [取り込み件数]');
    process.exit(1);
  }

  const inputPath = path.resolve(inputPathArg);
  const outputPath = path.resolve(outputPathArg);
  const limit = limitArg ? Number(limitArg) : Infinity;

  if (!fs.existsSync(inputPath)) {
    console.error(`入力ファイルが見つかりません: ${inputPath}`);
    process.exit(1);
  }

  console.log(`UniDic CSV を読み込み中: ${inputPath}`);
  if (Number.isFinite(limit)) {
    console.log(`取り込み上限: ${limit} 件`);
  }

  const entries = [];
  let lineNumber = 0;

  const stream = fs.createReadStream(inputPath, { encoding: 'utf8' });
  const rl = readline.createInterface({ input: stream, crlfDelay: Infinity });

  for await (const line of rl) {
    lineNumber += 1;
    if (lineNumber === 1) continue; // ヘッダーをスキップ
    if (!line) continue;
    if (entries.length >= limit) break;

    // UniDic の CSV はカンマ区切り、引用符で囲まれる場合もある
    // ここでは簡易的に split し、必要なら前後の引用符を除去する
    const cols = line.split(',').map((value) => value.replace(/^"(.*)"$/, '$1'));
    const surface = cols[0];
    if (!surface) continue;

    const lemma = cols[10] || surface;   // 見出し語
    const reading = cols[11] || surface; // 読み
    const pos = cols[4] || '名詞';       // 品詞大分類
    const subpos = cols[5] || '*';       // 品詞細分類

    entries.push({
      surface,
      reading,
      lemma,
      pos,
      subpos,
      cost: 5000
    });
  }

  const outputDir = path.dirname(outputPath);
  if (!fs.existsSync(outputDir)) {
    fs.mkdirSync(outputDir, { recursive: true });
  }

  const version = '2024.4.17';
  const isSubset = Number.isFinite(limit);
  const label = isSubset
    ? `UniDic ${version} subset (${entries.length} entries)`
    : `UniDic ${version} full`;

  const payload = {
    label,
    version,
    source: 'UniDic (National Institute for Japanese Language and Linguistics)',
    createdAt: new Date().toISOString(),
    entryCount: entries.length,
    entries
  };

  fs.writeFileSync(outputPath, JSON.stringify(payload, null, 2), 'utf8');

  console.log(`JSON 辞書を書き出しました: ${outputPath}`);
  console.log(`エントリ件数: ${entries.length}`);
  console.log('MorphFoundry の defaultDictionaries 設定でこの JSON をロードしてください。');
}

main().catch((error) => {
  console.error('変換中にエラーが発生しました:', error);
  process.exit(1);
});

